from sys import exit
from sys import exc_info
import thread
import time
import GPIO
import MDM
import MOD

try:
    from version import __version__
except ImportError:
    __version__ = "unknown"
    pass


class ModemCmd:
    """ Modem AT command strings.
    """

    def __init__(self):
        pass

    CGSN = 'AT+CGSN\r'
    CPIN = 'AT+CPIN?\r'
    CMGF = 'AT+CMGF=1\r'
    CNMI = 'AT+CNMI=2,1\r'
    CMGD = 'AT+CMGD='
    CFUN = '+CFUN'
    CREG = '+CREG'
    CMTI = '+CMTI'
    CMGL = '+CMGL'
    CMGR = '+CMGR'
    CMGS = '+CMGS'
    CMGW = '+CMGW'
    CMSS = '+CMSS'
    CSQ = '+CSQ'

# List of valid commands.
cmd_list = ['REL', 'LED', 'STAT', 'PWD', 'ALTON', 'ALTOFF', 'RESET']
# Local configuration parameters.
config = {}
# SMS meta data, parsed on incoming SMS messages.
sms_meta = {}
# Modem command strings.
mdm_cmd = ModemCmd()
# Modem registration status.
mdm_reg_status = ''
# Flag indicating message storage is not empty.
sms_pending = False


def get_modem_response(timeout):
    """ Get the modem response.

    :param timeout: timeout in seconds.
    :return: modem response or empty string.
    """
    rsp = ''
    start = time.time()
    while time.time() - start < timeout:
        rsp = rsp + MDM.read()
        # Break when response is received.
        if rsp:
            break
    # Calculate the response time.
    rspTime = time.time() - start
    rspTime = round(rspTime,2)
    if rspTime >= timeout - 0.5:
        print 'Timed out. Response time: ' + str(rspTime) + '\r'
    return rsp


def create_default_config():
    """ Create and store default configuration parameters.
    """
    cfg = {'rel_state': 'unknown', 'led_ctl': 'OFF', 'imei': 'unknown',
           'pwd': '0000', 'net': 'unknown', 'stat_ctl': 'OFF', 'alt_on': '', 'alt_off': '' }
    fo = open('config.txt', "w")
    fo.write(str(cfg))
    fo.close()


def read_config():
    """ Read configuration parameters into local variable.
    """
    global config
    try:
        fo = open('config.txt', "r")
        config = eval(fo.read())
        fo.close()
    except IOError:
        print 'Config file does not exist... creating.\r'
        create_default_config()
        fo = open('config.txt', "r")
        config = eval(fo.read())
        fo.close()


def write_config():
    """ Write configuration parameters to file.
    """
    global config
    if config:
        fo = open('config.txt', "w")
        fo.write(str(config))
        fo.close()


def get_stat_ctl():
    """ Get status report control parameter.

    :return: status report control parameter or empty string.
    """
    stat_ctl = ''
    global config
    if not config:
        read_config()
    if config and 'stat_ctl' in config:
        stat_ctl = config['stat_ctl']
    return stat_ctl


def get_relay_state():
    """ Get current relay state.

    :return: relay state or empty string.
    """
    relay_state = ''
    global config
    if not config:
        read_config()
    if config and 'rel_state' in config:
        relay_state = config['rel_state']
    return relay_state


def get_led_ctl():
    """ Get status LED control parameter.

    :return: status LED control parameter or empty string.
    """
    led_ctl = ''
    global config
    if not config:
        read_config()
    if config and 'led_ctl' in config:
        led_ctl = config['led_ctl']
    return led_ctl


def get_password():
    """ Get password.

    :return: password or empty string.
    """
    pwd = ''
    global config
    if not config:
        read_config()
    if config and 'pwd' in config:
        pwd = config['pwd']
    return pwd


def change_password(new_pwd):
    """ Change the existing password to the new password.

    :param new_pwd: new password.
    :return: True if the password was changed, False otherwise.
    """
    global config
    if not config:
        read_config()
    if config:
        if len(new_pwd) > 4:
            print 'password truncated\r'
            new_pwd = new_pwd[0:4]
        if len(new_pwd) == 4:
            print 'Changing password...\r'
            config['pwd'] = new_pwd
            write_config()
            return True
        else:
            print 'Password not changed - must be 4 characters\r'
            return False


def reset_password():
    """ Reset the password to 0000.
    """
    global config
    if not config:
        read_config()
    if config:
        config['pwd'] = '0000'
        write_config()


def get_input_voltage():
    """ Get the input voltage.

    :return: the input voltage in volts to 1 decimal place.
    """
    millivolts = float(GPIO.getADC(2)) * 1064.9 / 64.9
    # Convert to volts.
    volts = round(millivolts / 1000, 1)
    return volts


def get_switched_voltage():
    """ Get the switched output voltage.

    :return: the switched output voltage in volts to 1 decimal place.
    """
    millivolts = float(GPIO.getADC(1)) * 1064.9 / 64.9
    # Convert to volts.
    volts = round(millivolts / 1000, 1)
    return volts


def get_csq():
    """ Get the RSSI and BER.

    :return: an array of {RSSI, BER}
    """
    csq = {'rssi': 'unknown', 'ber': 'unknown'}
    csq_cmd = 'AT' + mdm_cmd.CSQ + '\r'
    MDM.send(csq_cmd, 0)
    # Get a string from the modem.
    rsp = get_modem_response(2)
    # A timeout results in a null string.
    if rsp:
        # Search for the CSQ response substring.
        index = rsp.find(mdm_cmd.CSQ)
        if index != -1:
            index = rsp.find(':', index)
            substring = rsp[index + 1:]
            substring = substring.strip()
            elements = substring.split(',')
            if len(elements) > 0:
                rssi = elements[0]
                csq['rssi'] = rssi
            if len(elements) > 1:
                ber = elements[1]
                csq['ber'] = ber
    return csq


def get_imei():
    """ Get the modem IMEI number.

    :return: the 15 character IMEI number as a string.
    """
    imei = 'unknown'
    MDM.send(mdm_cmd.CGSN, 0)
    # Get a string from the modem.
    rsp = get_modem_response(2)
    # A timeout results in a null string.
    if rsp:
        rsp = rsp.strip()
        elements = rsp.split('\r')
        imei = elements[0].strip()
        # Store the IMEI.
        if len(imei) == 15:
            global config
            if not config:
                read_config()
            if config:
                config['imei'] = imei
                write_config()
    return imei


def delete_sms(msg_idx):
    """ Delete the stored SMS message indicated by msg_idx.

    :param msg_idx: the index of the message to delete.
    """
    is_modem_ready()
    # Delete SMS message.
    cmd = mdm_cmd.CMGD + str(msg_idx) + '\r'
    print cmd
    MDM.send(cmd, 0)
    rsp = get_modem_response(10)
    print rsp


def delete_sms_all():
    """ Delete all stored SMS messages.
    """
    is_modem_ready()
    # Delete all SMS messages.
    cmd = mdm_cmd.CMGD + '1,4\r'
    print cmd
    MDM.send(cmd, 0)
    rsp = get_modem_response(20)
    print rsp


def send_message(msg):
    global sms_meta
    if sms_meta['ph_num']:
        write_sms(sms_meta['ph_num'], msg)


def send_status():
    """ Build a status report and write it to message storage.
    It will be sent from the manage_sms_storage routine.
    """
    # Build the message that will be returned.
    led_ctl = 'unknown'
    stat_ctl = 'unknown'
    imei = 'unknown'
    relay_state = 'unknown'
    alton = 'unknown'
    altoff = 'unknown'
    print 'Reading config file'
    global config
    if not config:
        read_config()
    if config:
        if 'led_ctl' in config:
            led_ctl = config['led_ctl']
        if 'stat_ctl' in config:
            stat_ctl = config['stat_ctl']
        if 'imei' in config:
            imei = config['imei']
        if 'rel_state' in config:
            relay_state = config['rel_state']
        if 'alt_on' in config:
            alton = config['alt_on']
        if 'alt_off' in config:
            altoff = config['alt_off']
    if imei == 'unknown':
        imei = get_imei()
    input_voltage = str(get_input_voltage())
    switched_voltage = str(get_switched_voltage())
    csq = get_csq()
    print 'Building message\r'
    ret_str = 'Ver: ' + __version__ + '\n'
    ret_str += 'IMEI: ' + imei + '\n'
    ret_str += 'Relay State: ' + relay_state + '\n'
    ret_str += 'Status LED: ' + led_ctl + '\n'
    ret_str += 'Status Report: ' + stat_ctl + '\n'
    ret_str += 'AltON: ' + alton + '\n'
    ret_str += 'AltOFF: ' + altoff + '\n'
    ret_str += 'RSSI: ' + csq['rssi'] + '\n'
    ret_str += 'Input: ' + input_voltage + 'V\n'
    ret_str += 'Output: ' + switched_voltage + 'V'
    print ret_str + '\r'
    print 'Message size: ' + str(len(ret_str)) + '\r'
    global sms_meta
    if sms_meta['ph_num']:
        write_sms(sms_meta['ph_num'], ret_str)


def set_relay():
    """ Set relay (closed circuit).
    """
    # Relay Set - GPIO 6 output, initially low.
    GPIO.setIOdir(6, 0, 1)
    # Relay Reset - GPIO 7 output, initially low.
    GPIO.setIOdir(7, 0, 1)
    # Toggle high.
    GPIO.setIOvalue(6, 1)
    time.sleep(1)
    # Toggle low.
    GPIO.setIOvalue(6, 0)
    # Update the configuration file.
    global config
    if not config:
        read_config()
    if config:
        config['rel_state'] = 'ON'
        write_config()


def reset_relay():
    """ Reset relay (open circuit).
    """
    # Relay Set - GPIO 6 output, initially low.
    GPIO.setIOdir(6, 0, 1)
    # Relay Reset - GPIO 7 output, initially low.
    GPIO.setIOdir(7, 0, 1)
    # Toggle high.
    GPIO.setIOvalue(7, 1)
    time.sleep(1)
    # Toggle low.
    GPIO.setIOvalue(7, 0)
    # Update the configuration file.
    global config
    if not config:
        read_config()
    if config:
        config['rel_state'] = 'OFF'
        write_config()


def set_alt_on_str(on_str):
    """ Set the string used as an alternative way to set the relay.
    If the string is found within the received message the relay
    will be set (ON).

    :param on_str: string used to set the relay.
    """
    global config
    if not config:
        read_config()
    if config:
        config['alt_on'] = on_str
        write_config()


def set_alt_off_str(off_str):
    """ Set the string used as an alternative way to reset the relay.
    If the string is found within the received message the relay
    will be reset (OFF).

    :param off_str: string used to reset the relay.
    """
    global config
    if not config:
        read_config()
    if config:
        config['alt_off'] = off_str
        write_config()


def led_delay_off():
    """ Delay by 60 seconds then disable the status LED.
    """
    time.sleep(60)
    GPIO.setIOdir(8, 0, 0)


def configure_status_led():
    """ Configure how the status LED should function.
    When the LED is OFF a thread is created that will turn the
    LED off in 60 seconds.
    When the LED is ON it will continuously flash network status.
    """
    if get_led_ctl() == 'OFF':
        print 'Status LED off in 60s\r'
        thread.start_new_thread(led_delay_off, ())
    elif get_led_ctl() == 'ON':
        print 'Status LED on\r'
        GPIO.setIOdir(8, 0, 2)


def enable_status_led():
    """ Enable the status LED and update the configuration file.
    """
    global config
    if not config:
        read_config()
    if config:
        config['led_ctl'] = 'ON'
        write_config()
    GPIO.setIOdir(8, 0, 2)


def disable_status_led():
    """ Disable the status LED and update the configuration file.
    """
    global config
    if not config:
        read_config()
    if config:
        config['led_ctl'] = 'OFF'
        write_config()
    GPIO.setIOdir(8, 0, 0)


def enable_status_report():
    """ Enable auto status reporting and update the configuration file.
    """
    global config
    if not config:
        read_config()
    if config:
        config['stat_ctl'] = 'ON'
        write_config()


def disable_status_report():
    """ Disable auto status reporting and update the configuration file.
    """
    global config
    if not config:
        read_config()
    if config:
        config['stat_ctl'] = 'OFF'
        write_config()


def get_modem_registration_status():
    """ Get registration status.

    :return: the registration status which may be:
    0 - not registered, ME is not currently searching a new operator to register to
    1 - registered, home network
    2 - not registered, but ME is currently searching a new operator to register to
    3 - registration denied
    4 -unknown
    5 - registered, roaming
    """
    global mdm_reg_status
    mdm_reg_status = ''
    MDM.send('AT' + mdm_cmd.CREG + '?\r', 0)
    # Get a string from the modem.
    rsp = get_modem_response(2)
    print rsp
    # A timeout results in a null string.
    if rsp:
        # Search for the CREG response substring.
        index = rsp.find(mdm_cmd.CREG)
        if index != -1:
            first = rsp.find(':', index)
            last = rsp.find('\r', index)
            substring = rsp[first + 1:last]
            substring = substring.strip()
            elements = substring.split(',')
            if len(elements) > 1:
                mdm_reg_status = elements[1]
    return mdm_reg_status


def is_modem_registered():
    """ Return True if the modem is registered on the network, False otherwise.

    :return: True or False.
    """
    global mdm_reg_status
    if mdm_reg_status == '1' or mdm_reg_status == '5':
        return True
    else:
        return False


def is_modem_ready():
    """ Return the number of seconds to respond to AT.
    If the modem does not respond a reset will be triggered
    by the watchdog timer.

    :return: seconds to respond.
    """
    count = 0
    while True:
        count += 1
        MDM.send('AT\r', 0)
        rsp = get_modem_response(2)
        if rsp.find('OK') != -1:
            break
    return count


def is_sim_ready():
    """ Return True if the SIM is ready, False otherwise.

    :return: True or False.
    """
    MDM.send(mdm_cmd.CPIN, 0)
    rsp = get_modem_response(2)
    if rsp.find('READY') != -1:
        return True
    else:
        return False


def is_imei_match(pwd):
    """ Return True if the last 6 digits of the IMEI match the given password, False otherwise.

    :return: True or False.
    """
    global config
    if not config:
        read_config()
    if config and 'imei' in config:
        imei = config['imei']
        imei = imei[-6:]
        if imei == pwd:
            return True
        else:
            return False
    else:
        return False


def is_password_match(pwd):
    """ Return True if the given password matches the stored password, False otherwise.

    :return: True or False.
    """
    global config
    if not config:
        read_config()
    if config:
        print 'Checking password...\r'
        if config['pwd'] == pwd:
            print 'Match\r'
            return True
        else:
            print 'No match\r'
            return False
    else:
        return False


def is_valid_command(cmd_str):
    """ Return True if cmd_str contains a valid command, False otherwise.

    :return: True or False.
    """
    elements = cmd_str.split(',')
    if len(elements) > 1:
        cmd = elements[1]
        cmd = cmd.upper()
        if cmd in cmd_list:
            return True
        else:
            return False
    else:
        return False


def write_sms(num, msg):
    """ Write SMS message to storage.

    :param num: the recipient phone number.
    :param msg: the message to send.
    """
    # Only 10 digit phone numbers are allowed.
    if len(num) != 10:
        print 'Phone number is not 10 digits\r'
        return
    if not msg:
        return
    MDM.send(mdm_cmd.CMGF, 0)
    get_modem_response(2)
    cmd = 'AT' + mdm_cmd.CMGW + '=' + num + '\r'
    print cmd
    MDM.send(cmd, 0)
    # The response should be >.
    rsp = get_modem_response(2)
    print rsp
    MDM.send(msg, 0)
    # Send termination character CTRL-Z.
    MDM.sendbyte(0x1A, 0)
    rsp = get_modem_response(10)
    print rsp
    if rsp.find('OK') != -1:
        global sms_pending
        sms_pending = True


def send_sms(num, msg):
    """ Send SMS message directly.

    :param num: the recipient phone number.
    :param msg: the message to send.
    """
    # Only 10 digit phone numbers are allowed.
    if len(num) != 10:
        return
    if not msg:
        return
    if not is_modem_registered():
        return
    MDM.send(mdm_cmd.CMGF, 0)
    rsp = get_modem_response(2)
    print rsp
    send_cmd = 'AT' + mdm_cmd.CMGS + '=' + num + '\r'
    print send_cmd
    MDM.send(send_cmd, 0)
    # The response should be >.
    rsp = get_modem_response(2)
    print rsp
    MDM.send(msg, 0)
    # Send termination character CTRL-Z.
    MDM.sendbyte(0x1A, 0)
    rsp = get_modem_response(10)
    print rsp


def list_sms(list_type):
    """ List stored SMS messages.

    :param list_type: the type of messages to list which may be:
    "REC UNREAD" - new message
    "REC READ" - read message
    "STO UNSENT" - stored message not yet sent
    "STO SENT" - stored message already sent
    "ALL" - all messages.
    :return: the list of messages as a string.
    """
    MDM.send(mdm_cmd.CMGF, 0)
    get_modem_response(2)
    rsp = ''
    for x in xrange(5):
        cmd = 'AT' + mdm_cmd.CMGL + '="' + list_type + '"\r'
        MDM.send(cmd, 0)
        rsp = get_modem_response(10)
        if rsp.find('ERROR') != -1:
            continue
        else:
            break
    # Some listings may be long.
    # Use this command to wait for the modem.
    is_modem_ready()
    return rsp


def initialize_modem():
    """ Initialize the modem configuration.
    """
    # Enable the status LED.
    GPIO.setIOdir(8, 0, 2)
    # The modem must respond to AT.
    seconds = is_modem_ready()
    print 'modem ready in ' + str(seconds) + ' seconds\r'
    # A SIM card must be detected in order to send/receive SMS message.
    if is_sim_ready():
        print 'SIM ready\r'
    else:
        print 'SIM not detected\r'
    # Print the IMEI number for reference.
    imei = get_imei()
    print 'IMEI: ' + imei + '\r'
    # Initialize the modem configuration.
    print 'Initializing modem...\r'
    # Relay Set - GPIO 6 output, initially low.
    result = GPIO.setIOdir(6, 0, 1)
    if result == 1:
        print 'Relay SET is output, low\r'
    else:
        print 'Relay SET GPIO error\r'
    # Relay Reset - GPIO 7 output, initially low.
    result = GPIO.setIOdir(7, 0, 1)
    if result == 1:
        print 'Relay RESET is output, low\r'
    else:
        print 'Relay RESET GPIO error\r'
    # Select text mode for SMS messages.
    MDM.send(mdm_cmd.CMGF, 0)
    rsp = get_modem_response(2)
    if rsp.find('OK') != -1:
        print 'SMS text mode selected\r'
    else:
        print 'Error setting SMS text mode\r'
    # Buffer unsolicited code and SMS memory location routed to the TE.
    MDM.send(mdm_cmd.CNMI, 0)
    rsp = get_modem_response(2)
    if rsp.find('OK') != -1:
        print 'SMS unsolicited code routed to TE\r'
    else:
        print 'Error setting SMS unsolicited code\r'
    # Set power saving mode.
    cmd = 'AT' + mdm_cmd.CFUN + '=' + str(5) + '\r'
    MDM.send(cmd, 0)
    rsp = get_modem_response(2)
    if rsp.find('OK') != -1:
        print 'Power saving mode selected\r'
    else:
        print 'Error setting power mode\r'


def run_command(cmd, action):
    """ Run the given command.

    :param cmd: the command to run.
    :param action: the optional action to complement the command.
    """
    print 'Running command...\r'
    stat_ctl = get_stat_ctl()
    cmd = cmd.upper()
    if cmd <> 'ALTON' and cmd <> 'ALTOFF':
        action = action.upper()
    if cmd == 'REL':
        if action == 'ON':
            print 'Set relay\r'
            set_relay()
        elif action == 'OFF':
            print 'Reset relay\r'
            reset_relay()
    elif cmd == 'LED':
        if action == 'ON':
            print 'Enable status LED\r'
            enable_status_led()
        elif action == 'OFF':
            print 'Disable status LED\r'
            disable_status_led()
    elif cmd == 'PWD':
        print 'Change password.\r'
        result = change_password(action)
        # Do not send status for this command.
        stat_ctl = 'OFF'
        if result == True:
            pwd = get_password()
            send_message('Pass code changed to: ' + pwd)
        else:
            send_message('Error: pass code not changed - must be 4 characters')
    elif cmd == 'STAT':
        if action == 'ON':
            print 'Enable status response\r'
            enable_status_report()
            send_status()
        elif action == 'OFF':
            print 'Disable status response\r'
            disable_status_report()
        elif action == '' and stat_ctl == 'OFF':
            print 'Get status\r'
            send_status()
    elif cmd == 'ALTON':
        print 'Setting alternative ON string\r'
        set_alt_on_str(action)
    elif cmd == 'ALTOFF':
        print 'Setting alternative OFF string\r'
        set_alt_off_str(action)
    # Send auto status report if configured.
    if stat_ctl == 'ON':
        send_status()


def process_rx_meta(meta_str):
    """ Process the meta portion of the SMS message.

    :param meta_str: the meta data as a string.
    :return:
    """
    if not meta_str:
        return
    global sms_meta
    elements = meta_str.split(',')
    if len(elements) > 0:
        status = elements[0]
        sms_meta['status'] = status.strip('"')
    if len(elements) > 1:
        ph_num = elements[1]
        ph_num = ph_num.strip('"')
        if len(ph_num) > 10:
            print 'Phone num truncated\r'
            ph_num = ph_num[-10:]
            print 'Num: ' + ph_num + '\r'
        sms_meta['ph_num'] = ph_num
    if len(elements) > 3:
        date = elements[3]
        sms_meta['date'] = date.strip('"')
    if len(elements) > 4:
        sms_time = elements[4]
        sms_meta['time'] = sms_time.strip('"')


def search_alt_str(body_str):
    """ Search body_str for the ALTON or ALTOFF string.
    If an alternative string is found a new cmd_str is created and returned.

    :return: The new cmd_str or body_str if no alternative string is found.
    """
    global config
    if not config:
        read_config()
    if config:
        if 'alt_on' in config:
            on_str = config['alt_on']
        if 'alt_off' in config:
            off_str = config['alt_off']
    if on_str:
        index = body_str.find(on_str,5)
        if index != -1:
            print 'Alternative ON string found\r'
            cmd_str = body_str[:5]
            cmd_str += 'REL,ON'
            return cmd_str
    if off_str:
        index = body_str.find(off_str,5)
        if index != -1:
            print 'Alternative OFF string found\r'
            cmd_str = body_str[:5]
            cmd_str += 'REL,OFF'
            return cmd_str
    return body_str


def process_rx_cmd(cmd_str):
    """ Process the body of the SMS message as a command string.

    :param cmd_str: the command string.
    :return: True if the command was a RESET, False otherwise.
    """
    if not cmd_str:
        print 'Missing command string\r'
        return False

    if len(cmd_str) < 6:
        print 'Command string too short\r'
        return False

    # An underscore is an optional character to separate the password.
    if cmd_str[4] == '_':
        tmp_str = cmd_str[:4]
        tmp_str += ','
        tmp_str += cmd_str[5:]
        cmd_str = tmp_str
    # If there is no command, check for ALTON or ALTOFF strings.
    if not is_valid_command(cmd_str):
        print 'Missing valid command\r'
        cmd_str = search_alt_str(cmd_str)
    # If the resulting string does not contain a command, return.
    if not is_valid_command(cmd_str):
        print 'Returning\r'
        return False

    print 'Processing command... ' + cmd_str + '\r'
    cmd = ''
    action = ''
    elements = cmd_str.split(',')
    if len(elements) > 0:
        pwd = elements[0]
        if len(elements) > 1:
            cmd = elements[1]
            cmd = cmd.upper()
            print 'Command: ' + cmd + '\r'
        if len(elements) > 2:
            action = elements[2]
            action = action.upper()
            if cmd == 'ALTON' or cmd == 'ALTOFF':
                # Get the substring following the second comma.
                index = cmd_str.find(',')
                index = cmd_str.find(',', index+1)
                action = cmd_str[index+1:]
            print 'Action: ' + action + '\r'
        # Check the password.
        if is_password_match(pwd):
            if cmd == 'RESET':
                return True
            run_command(cmd, action)
        # Password reset command.
        elif cmd == 'PWD' and action == 'RESET':
            if is_imei_match(pwd):
                reset_password()
                send_message('Pass code reset to 0000')
    return False


def parse_rx_sms(rsp):
    """ Parse the SMS message into meta + body.

    :param rsp: the SMS message to parse.
    :return: an array of {META,BODY} or nothing.
    """
    if not rsp:
        return
    print 'Parsing: ' + rsp
    # Search for the CMGR response substring.
    index = rsp.find(mdm_cmd.CMGR)
    if index != -1:
        sms = {}
        index = rsp.find(':', index)
        substring = rsp[index + 1:]
        substring = substring.strip()
        elements = substring.split('\r')
        if len(elements) > 0:
            meta = elements[0]
            meta = meta.strip()
            print 'Meta: ' + meta + '\r'
            if len(elements) > 1:
                body = elements[1]
                body = body.strip()
                print 'Body: ' + body + '\r'
                sms['meta'] = meta
                sms['body'] = body
        return sms


def process_creg(rsp):
    """ Process the response from the CREG command.

    :param rsp: the modem response to process.
    """
    index = rsp.find(mdm_cmd.CREG)
    index = rsp.find(':', index)
    substring = rsp[index + 1:]
    status = substring.strip()
    if status == '0':
        print 'not registered\r'
    if status == '1':
        print 'registered, home network\r'
    if status == '2':
        print 'searching\r'
    if status == '3':
        print 'registration denied\r'
    if status == '4':
        print 'unknown\r'
    if status == '5':
        print 'registered, roaming\r'


def process_cmgr(msg_idx):
    """ Read a message from storage using the CMGR command.

    :param msg_idx: the index of the message to read from storage.
    """
    reset_pending = False
    read_cmd = 'AT' + mdm_cmd.CMGR + '=' + str(msg_idx) + '\r'
    print read_cmd
    MDM.send(read_cmd, 0)
    # Read the complete SMS message.
    sms_msg = get_modem_response(10)
    print sms_msg
    if sms_msg:
        sms = parse_rx_sms(sms_msg)
        if sms:
            process_rx_meta(sms['meta'])
            reset_pending = process_rx_cmd(sms['body'])
    # Delete the SMS message.
    time.sleep(2)
    delete_sms(msg_idx)
    time.sleep(2)
    if reset_pending:
        # Perform a reset by forcing a watchdog timeout.
        MOD.watchdogEnable(1)
        time.sleep(5)


def process_cmss(msg_idx):
    """ Send a message from storage using the CMSS command.

    :param msg_idx: the index of the message to send from storage.
    """
    cmd = 'AT' + mdm_cmd.CMSS + '=' + str(msg_idx) + '\r'
    print cmd
    MDM.send(cmd, 0)
    rsp = get_modem_response(10)
    print rsp
    is_modem_ready()


def process_cmgd(msg_idx):
    """ Delete a message from storage using the CMGD command.

    :param msg_idx: the index of the message to delete from storage.
    """
    # Delete the SMS message.
    delete_sms(msg_idx)
    time.sleep(2)


def manage_sms_storage():
    """ List stored messages and process them in order.
    """
    is_modem_ready()
    msg_idx = ''
    # List all read messages.
    rsp = list_sms('ALL')
    if not rsp:
        return
    index = rsp.find(mdm_cmd.CMGL)
    if index != -1:
        global sms_pending
        sms_pending = True
        index = rsp.find(':', index)
        substring = rsp[index + 1:]
        substring = substring.strip()
        elements = substring.split(',')
        if len(elements) > 0:
            msg_idx = elements[0]
        if len(elements) > 1:
            status = elements[1]
            if status.find('READ') != -1:
                if msg_idx:
                    process_cmgr(msg_idx)
            elif status.find('STO UNSENT') != -1:
                if msg_idx:
                    process_cmss(msg_idx)
            elif status.find('STO SENT') != -1:
                if msg_idx:
                    process_cmgd(msg_idx)
    else:
        # If there are no messages left, sms_pending should be False.
        if sms_pending:
            print 'resetting sms_pending flag\r'
            sms_pending = False


def reset_watchdog():
    """ Reset the watchdog timer.
    """
    print 'watchdog reset\r'
    MOD.watchdogReset()


def start_watchdog():
    """ Enable the watchdog timer at 60 second intervals.
    """
    MOD.watchdogEnable(60)


def task_loop():
    # Setup functions run once.
    start_watchdog()
    initialize_modem()
    configure_status_led()
    # Enter the main task loop.
    while True:
        try:
            while True:
                # Reset the watchdog timer.
                reset_watchdog()
                # Clean up stored SMS messages.
                manage_sms_storage()
                # If a message is pending bypass sleep.
                global sms_pending
                if sms_pending == False:
                    MOD.powerSaving(25)
        except:
            # Get trace information
            type, value, tb = exc_info()
            print 'Exception :', type, '=', value, '\r'
            while tb is not None:
                print 'File "%s", %d in %s \r' % (tb.tb_frame.f_code.co_filename, tb.tb_lineno, tb.tb_frame.f_code.co_name)
                tb = tb.tb_next
            # Delete all SMS messages.
            # If a message caused the exception it could cause other messages not to be read.
            delete_sms_all()
            time.sleep(5)
            pass
